<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Models\Amphur;
use App\Http\Models\Electorate;
use App\Http\Models\Home;
use App\Http\Models\Mooban;
use App\Http\Models\Tambon;
use App\Http\Models\User;
use Illuminate\Http\Request;
use Exception;

class HomesController extends Controller
{

    /**
     * Display a listing of the homes.
     *
     * @return Illuminate\View\View
     */
    public function index()
    {
        $homes = Home::with('electorate', 'amphur', 'tambon', 'mooban', 'user')->paginate(25);

        return view('homes.index', compact('homes'));
    }

    /**
     * Show the form for creating a new home.
     *
     * @return Illuminate\View\View
     */
    public function create()
    {
        $electorates = Electorate::pluck('name', 'id')->all();
        $amphurs = Amphur::pluck('name_th', 'id')->all();
        $moobans = Mooban::orderbY('id','asc')->get();
        $tambons = Tambon::pluck('name', 'id')->all();
        $users = User::pluck('name', 'id')->all();

        return view('homes.create', compact('electorates', 'amphurs', 'tambons', 'moobans', 'users'));
    }

    /**
     * Store a new home in the storage.
     *
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        try {

            $home = Home::where('moobans_id', $request->moobans_id)->where('address', $request->input('address'))->count();
            if ($home > 0) {
                return back()->withInput()
                    ->withErrors(['unexpected_error' => 'มีบ้านเลขที่นี้ในหมู่บ้านนีแล้ว!']);
            }
            $data = $this->getData($request);


            Home::create($data);

            return redirect()->route('homes.home.index')
                ->with('success_message', 'เพิ่ม Home ใหม่แล้ว');
        } catch (Exception $exception) {

            return back()->withInput()
                ->withErrors(['unexpected_error' => 'พบข้อผิดพลาดค่ะ!']);
        }
    }

    /**
     * Display the specified home.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function show($id)
    {
        $home = Home::with('electorate', 'amphur', 'tambon', 'mooban', 'user')->findOrFail($id);

        return view('homes.show', compact('home'));
    }

    /**
     * Show the form for editing the specified home.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function edit($id)
    {
        $home = Home::findOrFail($id);
        $electorates = Electorate::pluck('name', 'id')->all();
        $amphurs = Amphur::pluck('name_th', 'id')->all();
        $tambons = Tambon::pluck('name', 'id')->all();
        $moobans = Mooban::orderBy('id', 'asc')->get();

        $users = User::pluck('name', 'id')->all();

        return view('homes.edit', compact('home', 'electorates', 'amphurs', 'tambons', 'moobans', 'users'));
    }

    /**
     * Update the specified home in the storage.
     *
     * @param int $id
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function update($id, Request $request)
    {
        try {

            $data = $this->getData($request);

            $home = Home::findOrFail($id);
            $home->update($data);

            return redirect()->route('homes.home.index')
                ->with('success_message', 'ปรับปรุงข้อมูล Home เรียบร้อยแล้วค่ะ');
        } catch (Exception $exception) {

            return back()->withInput()
                ->withErrors(['unexpected_error' => 'พบข้อผิดพลาดค่ะ!']);
        }
    }

    /**
     * Remove the specified home from the storage.
     *
     * @param int $id
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function destroy($id)
    {
        try {
            $home = Home::findOrFail($id);
            $home->delete();

            return redirect()->route('homes.home.index')
                ->with('success_message', 'ลบข้อมูล Home เรียบร้อยแล้วค่ะ');
        } catch (Exception $exception) {

            return back()->withInput()
                ->withErrors(['unexpected_error' => 'พบข้อผิดพลาดค่ะ!']);
        }
    }


    /**
     * Get the request's data from the request.
     *
     * @param Illuminate\Http\Request\Request $request 
     * @return array
     */
    protected function getData(Request $request)
    {
        $rules = [
            'electorates_id' => 'nullable',
            'amphurs_id' => 'nullable',
            'tambons_id' => 'nullable',
            'moobans_id' => 'nullable',
            'address' => 'nullable|string|min:0|max:10',
            'users_id' => 'nullable',
            'remark' => 'nullable|string|min:0|max:150',
        ];

        $data = $request->validate($rules);


        return $data;
    }


    public function get_home_by_mooban($id)
    {
        $homes = Home::where('moobans_id', $id)->get();
        return view('homes.homes_select', compact('homes'));

    }


}
