<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Http\Models\Amphur;
use App\Http\Models\Electorate;
use App\Http\Models\Tambon;
use Illuminate\Http\Request;
use Exception;

class TambonsController extends Controller
{

    /**
     * Display a listing of the tambons.
     *
     * @return Illuminate\View\View
     */
    public function index()
    {
        $tambons = Tambon::with('amphur', 'electorate')->paginate(25);

        return view('tambons.index', compact('tambons'));
    }

    /**
     * Show the form for creating a new tambon.
     *
     * @return Illuminate\View\View
     */
    public function create()
    {
        $amphurs = Amphur::pluck('name_th', 'id')->all();
        $electorates = Electorate::pluck('name', 'id')->all();

        return view('tambons.create', compact('amphurs', 'electorates'));
    }

    /**
     * Store a new tambon in the storage.
     *
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        try {

            $data = $this->getData($request);

            Tambon::create($data);

            return redirect()->route('tambons.tambon.index')
                ->with('success_message', 'เพิ่ม Tambon ใหม่แล้ว');
        } catch (Exception $exception) {

            return back()->withInput()
                ->withErrors(['unexpected_error' => 'พบข้อผิดพลาดค่ะ!']);
        }
    }

    /**
     * Display the specified tambon.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function show($id)
    {
        $tambon = Tambon::with('amphur', 'electorate')->findOrFail($id);

        return view('tambons.show', compact('tambon'));
    }

    /**
     * Show the form for editing the specified tambon.
     *
     * @param int $id
     *
     * @return Illuminate\View\View
     */
    public function edit($id)
    {
        $tambon = Tambon::findOrFail($id);
        $amphurs = Amphur::pluck('name_th', 'id')->all();
        $electorates = Electorate::pluck('name', 'id')->all();

        return view('tambons.edit', compact('tambon', 'amphurs', 'electorates'));
    }

    /**
     * Update the specified tambon in the storage.
     *
     * @param int $id
     * @param Illuminate\Http\Request $request
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function update($id, Request $request)
    {
        try {

            $data = $this->getData($request);

            $tambon = Tambon::findOrFail($id);
            $tambon->update($data);

            return redirect()->route('tambons.tambon.index')
                ->with('success_message', 'ปรับปรุงข้อมูล Tambon เรียบร้อยแล้วค่ะ');
        } catch (Exception $exception) {

            return back()->withInput()
                ->withErrors(['unexpected_error' => 'พบข้อผิดพลาดค่ะ!']);
        }
    }

    /**
     * Remove the specified tambon from the storage.
     *
     * @param int $id
     *
     * @return Illuminate\Http\RedirectResponse | Illuminate\Routing\Redirector
     */
    public function destroy($id)
    {
        try {
            $tambon = Tambon::findOrFail($id);
            $tambon->delete();

            return redirect()->route('tambons.tambon.index')
                ->with('success_message', 'ลบข้อมูล Tambon เรียบร้อยแล้วค่ะ');
        } catch (Exception $exception) {

            return back()->withInput()
                ->withErrors(['unexpected_error' => 'พบข้อผิดพลาดค่ะ!']);
        }
    }


    /**
     * Get the request's data from the request.
     *
     * @param Illuminate\Http\Request\Request $request 
     * @return array
     */
    protected function getData(Request $request)
    {
        $rules = [
            'amphurs_id' => 'required',
            'name' => 'nullable|string|min:0|max:75',
            'electorates_id' => 'required',
        ];

        $data = $request->validate($rules);


        return $data;
    }



    public function get_tambon_by_amphur($id)
    {
        $tambons = Tambon::where('amphurs_id', $id)->get();
        return view('tambons.tambons_select2', compact('tambons'));

    }
}
